import datetime
import os
import time
import torch
from torch.utils.data import DataLoader
import torch.nn as nn
import torch.nn.functional as F
from torch.utils.tensorboard import SummaryWriter
from torch.cuda import amp
from models import optimal_model

import argparse
from spikingjelly.clock_driven import functional
from spikingjelly.clock_driven import surrogate as surrogate_sj

from utils import Bar, Logger, AverageMeter, accuracy
import torch.utils.data as data
import torchvision.transforms as transforms
import torchvision.datasets as datasets
from torchtoolbox.transform import Cutout
from utils.cifar10_dvs import CIFAR10DVS, ToPILImage, Resize, ToTensor
from spikingjelly.datasets.dvs128_gesture import DVS128Gesture
import collections
import random
import numpy as np
torch.backends.cudnn.deterministic = True
torch.backends.cudnn.benchmark = False

'''         
                                   
        This script is used to retrain the optimal neuron configuration obtained by train_search.py
        
        Get started through:
        
        python train_optimal.py -dataset 'cifar10' -model 'spiking_resnet18' -optim_structure 'xxx';
        
'''

def main():
    parser = argparse.ArgumentParser(description='Retraining')
    parser.add_argument('-seed', default=2024, type=int, help='hope you have good luck in the year of 2024 :D')
    parser.add_argument('-name', default='', type=str, help='')
    parser.add_argument('-T', default=6, type=int, help='time step setting')
    parser.add_argument('-tau', default=1.1, type=float, help='membrane time constant')
    parser.add_argument('-b', default=128, type=int, help='batchsize')
    parser.add_argument('-epochs', default=300, type=int, metavar='N', help='epoch')
    parser.add_argument('-T_max', default=300, type=int, help='T_max for CosineAnnealingLR')
    parser.add_argument('-dataset', default='cifar100', type=str, help='name of the dataset, cifar10, cifar100, tinyimagenet dvscifar10, or dvsgesture')
    parser.add_argument('-weight_decay', type=float, default=5e-4)
    parser.add_argument('-model', type=str, default='spiking_resnet18', help='network model')
    parser.add_argument('-j', default=4, type=int, metavar='N', help='number of loading workers')
    parser.add_argument('-data_dir', type=str, default='./data', help='directory of the dataset')
    parser.add_argument('-out_dir', type=str, default='./logs', help='root dir for saving and checkpoint')
    parser.add_argument('-surrogate', default='triangle', type=str, help='surrogate function')
    parser.add_argument('-resume', type=str, help='resume path')
    parser.add_argument('-amp', action='store_false', help='automatic mixed precision training')
    parser.add_argument('-opt', type=str, help='optimizer SGD or AdamW', default='SGD')
    parser.add_argument('-lr', default=0.1, type=float, help='learning rate')
    parser.add_argument('-momentum', default=0.9, type=float, help='momentum for SGD')
    parser.add_argument('-lr_scheduler', default='CosALR', type=str, help='scheduler')
    parser.add_argument('-step_size', default=100, type=float, help='step size for StepLR')
    parser.add_argument('-gamma', default=0.1, type=float, help='gamma for StepLR')
    parser.add_argument('-drop_rate', type=float, default=0.0, help='dropout rate')
    parser.add_argument('-loss_lambda', type=float, default=0.05, help='constant for TET loss')
    parser.add_argument('-mse_n_reg', action='store_true', help='loss function setting')
    parser.add_argument('-loss_means', type=float, default=1.0, help='used in the loss function when mse_n_reg=False')
    parser.add_argument('-save_init', action='store_true', help='save the initialization of parameters')
    parser.add_argument('-optim_structure', type=str, default='', help='MANDATORY REQUIRED: the optimal structure found by train_search.py,'
                                                                       ' a pth file with absolute path in your PC, if you dont have one, please run train_search.py first，'
                                                                       'or you can specify the optimal structure manually like 201020101010')
    args = parser.parse_args()
    print(args)
    _seed_ = args.seed
    random.seed(_seed_)
    torch.manual_seed(_seed_) 
    torch.cuda.manual_seed_all(_seed_)
    np.random.seed(_seed_)

    # data processing & network construct
    data_dir = args.data_dir
    if args.dataset == 'cifar10' or args.dataset == 'cifar100':
        c_in = 3
        if args.dataset == 'cifar10':
            dataloader = datasets.CIFAR10
            num_classes = 10
            normalization_mean = (0.4914, 0.4822, 0.4465)
            normalization_std = (0.2023, 0.1994, 0.2010)
        elif args.dataset == 'cifar100':
            dataloader = datasets.CIFAR100
            num_classes = 100
            normalization_mean = (0.5071, 0.4867, 0.4408)
            normalization_std = (0.2675, 0.2565, 0.2761)
        transform_train = transforms.Compose([
            transforms.RandomCrop(32, padding=4),
            Cutout(),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            transforms.Normalize(normalization_mean, normalization_std),
        ])
        transform_test = transforms.Compose([
            transforms.ToTensor(),
            transforms.Normalize(normalization_mean, normalization_std),
        ])
        trainset = dataloader(root=data_dir, train=True, download=True, transform=transform_train)
        train_data_loader = data.DataLoader(trainset, batch_size=args.b, shuffle=True, num_workers=args.j)

        testset = dataloader(root=data_dir, train=False, download=False, transform=transform_test)
        test_data_loader = data.DataLoader(testset, batch_size=args.b, shuffle=False, num_workers=args.j)
    elif args.dataset == 'dvscifar10':
        c_in = 2
        num_classes = 10
        data_dir = os.path.join(data_dir, 'dvscifar10')
        transform_train = transforms.Compose([
            ToPILImage(),
            Resize(48),
            ToTensor(),
        ])
        transform_test = transforms.Compose([
            ToPILImage(),
            Resize(48),
            ToTensor(),
        ])
        trainset = CIFAR10DVS(data_dir, train=True, use_frame=True, frames_num=args.T, split_by='number', normalization=None, transform=transform_train)
        train_data_loader = data.DataLoader(trainset, batch_size=args.b, shuffle=True, num_workers=args.j)
        testset = CIFAR10DVS(data_dir, train=False, use_frame=True, frames_num=args.T, split_by='number', normalization=None, transform=transform_test)
        test_data_loader = data.DataLoader(testset, batch_size=args.b, shuffle=False, num_workers=args.j)
    elif args.dataset == 'dvsgesture':
        c_in = 2
        num_classes = 11
        data_dir = os.path.join(data_dir, 'dvsgesture')
        trainset = DVS128Gesture(root=data_dir, train=True, data_type='frame', frames_number=args.T, split_by='number')
        train_data_loader = data.DataLoader(trainset, batch_size=args.b, shuffle=True, num_workers=args.j, drop_last=True, pin_memory=True)
        testset = DVS128Gesture(root=data_dir, train=False, data_type='frame', frames_number=args.T, split_by='number')
        test_data_loader = data.DataLoader(testset, batch_size=args.b, shuffle=False, num_workers=args.j, drop_last=False, pin_memory=True)
    elif args.dataset == 'tinyimagenet':
        c_in = 3
        data_dir = os.path.join(data_dir, 'tiny-imagenet-200')
        num_classes = 200
        traindir = os.path.join(data_dir, 'train')
        testdir = os.path.join(data_dir, 'val')
        normalize = transforms.Normalize(mean=[0.4802, 0.4481, 0.3975],
                                         std=[0.2770, 0.2691, 0.2821])
        transform_train = transforms.Compose([
            transforms.RandomCrop(64, padding=4),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            normalize,
        ])
        transform_test = transforms.Compose([
            transforms.Resize(64),
            transforms.ToTensor(),
            normalize,
        ])
        train_dataset = datasets.ImageFolder(traindir, transform=transform_train)
        train_data_loader = DataLoader(train_dataset, batch_size=args.b, shuffle=True, num_workers=args.j, pin_memory=True)
        test_dataset = datasets.ImageFolder(testdir, transform=transform_test)
        test_data_loader = DataLoader(test_dataset, batch_size=args.b, shuffle=False, num_workers=args.j, pin_memory=True)
    else:
        raise NotImplementedError

    # optimal neuron configuration preparing
    if args.optim_structure == '':
        raise ValueError('Please specify the optimal structure found by train_search.py, if you dont have one, please run train_search.py first')
    elif os.path.exists(args.optim_structure) and args.optim_structure.endswith('.pth'):
        optim_path = args.optim_structure
        optimal_structure = torch.load(optim_path)
        alphas = optimal_structure['super_net']['_alphas_archi']
        optimal_neuron = alphas.argmax(dim=1)
        print(f"Optimal Architecture provided by pth file: {optim_path}")
        print(f"Optimal Architecture: {optimal_neuron}")
    else:
        int_list = [int(char) for char in args.optim_structure]
        optimal_neuron = torch.tensor(int_list)
        print(f"Optimal Architecture provided manually")
        print(f"Optimal Architecture: {optimal_neuron}")
    optimal_str = "".join(map(str, optimal_neuron.tolist()))
    net = optimal_model.__dict__[args.model](optimal_neuron=optimal_neuron, num_classes=num_classes, neuron_dropout=args.drop_rate,
                                                      tau=args.tau, c_in=c_in)
    print('Total Parameters: %.2fM' % (sum(p.numel() for p in net.parameters()) / 1000000.0))
    net.cuda()

    # optimizer
    if args.opt == 'SGD':
        optimizer = torch.optim.SGD(net.parameters(), lr=args.lr, momentum=args.momentum, weight_decay=args.weight_decay)
    elif args.opt == 'AdamW':
        optimizer = torch.optim.AdamW(net.parameters(), lr=args.lr, weight_decay=args.weight_decay)
    else:
        raise NotImplementedError(args.opt)
    if args.lr_scheduler == 'StepLR':
        lr_scheduler = torch.optim.lr_scheduler.StepLR(optimizer, step_size=args.step_size, gamma=args.gamma)
    elif args.lr_scheduler == 'CosALR':
        lr_scheduler = torch.optim.lr_scheduler.CosineAnnealingLR(optimizer, T_max=args.T_max)
    else:
        raise NotImplementedError(args.lr_scheduler)
    scaler = None
    if args.amp:
        scaler = amp.GradScaler()

    # resume
    start_epoch = 0
    max_test_acc = 0
    if args.resume:
        print('resuming...')
        checkpoint = torch.load(args.resume, map_location='cpu')
        net.load_state_dict(checkpoint['net'])
        optimizer.load_state_dict(checkpoint['optimizer'])
        lr_scheduler.load_state_dict(checkpoint['lr_scheduler'])
        start_epoch = checkpoint['epoch'] + 1
        max_test_acc = checkpoint['max_test_acc']
        print('start epoch:', start_epoch, ', max test acc:', max_test_acc)
    if args.pre_train:
        checkpoint = torch.load(args.pre_train, map_location='cpu')
        state_dict2 = collections.OrderedDict([(k, v) for k, v in checkpoint['net'].items()])
        net.load_state_dict(state_dict2)
        print('use pre-trained model, max test acc:', checkpoint['max_test_acc'])

    # output setting
    out_dir = os.path.join(args.out_dir, f'Optimal_{args.dataset}_{args.model}_T{args.T}_tau{args.tau}_e{args.epochs}_bs{args.b}_{args.opt}_lr{args.lr}_wd{args.weight_decay}_drop{args.drop_rate}_losslamb{args.loss_lambda}_')
    if args.lr_scheduler == 'CosALR':
        out_dir += f'CosALR_{args.T_max}'
    elif args.lr_scheduler == 'StepLR':
        out_dir += f'StepLR_{args.step_size}_{args.gamma}'
    else:
        raise NotImplementedError(args.lr_scheduler)
    out_dir += f'_{optimal_str}'
    if not os.path.exists(out_dir):
        os.makedirs(out_dir)
        print(f'Mkdir {out_dir}.')
    else:
        print('out dir already exists:', out_dir)
    if args.save_init:
        checkpoint = {
            'net': net.state_dict(),
            'epoch': 0,
            'max_test_acc': 0.0
        }
        torch.save(checkpoint, os.path.join(out_dir, 'checkpoint_0.pth'))
    with open(os.path.join(out_dir, 'args.txt'), 'w', encoding='utf-8') as args_txt:
        args_txt.write(str(args))
    writer = SummaryWriter(os.path.join(out_dir, 'logs'), purge_step=start_epoch)

    # retraining
    criterion_mse = nn.MSELoss()
    for epoch in range(start_epoch, args.epochs):
        start_time = time.time()
        net.train()
        batch_time = AverageMeter()
        data_time = AverageMeter()
        losses = AverageMeter()
        top1 = AverageMeter()
        top5 = AverageMeter()
        end = time.time()
        bar = Bar('Retraining', max=len(train_data_loader))
        train_loss = 0
        train_acc = 0
        train_samples = 0
        batch_idx = 0
        for frame, label in train_data_loader:
            batch_idx += 1
            if args.dataset != 'dvscifar10':
                frame = frame.float().cuda()
                if args.dataset == 'dvsgesture':
                    frame = frame.transpose(0, 1)
            t_step = args.T
            label = label.cuda()
            batch_loss = 0
            optimizer.zero_grad()
            for t in range(t_step):
                if args.dataset == 'dvscifar10':
                    input = frame[t].float().cuda()
                elif args.dataset == 'dvsgesture':
                    input = frame[t]
                else:
                    input = frame
                if args.amp:
                    with amp.autocast():
                        if t == 0:
                            output = net(input)
                            total_output = output.clone().detach()
                        else:
                            output = net(input)
                            total_output += output.clone().detach()
                        if args.loss_lambda > 0.0:
                            if args.mse_n_reg:
                                label_one_hot = F.one_hot(label, num_classes).float()
                            else:
                                label_one_hot = torch.zeros_like(output).fill_(args.loss_means).to(output.device)
                            mse_loss = criterion_mse(output, label_one_hot)
                            loss = ((1 - args.loss_lambda) * F.cross_entropy(output, label) + args.loss_lambda * mse_loss) / t_step
                        else:
                            loss = F.cross_entropy(output, label) / t_step
                    scaler.scale(loss).backward()
                    batch_loss += loss.item()
                    train_loss += loss.item() * label.numel()
                else:
                    raise NotImplementedError('Please use amp.')
            if args.amp:
                scaler.step(optimizer)
                scaler.update()
            else:
                optimizer.step()

            # accuracy and loss
            prec1, prec5 = accuracy(total_output.data, label.data, topk=(1, 5))
            losses.update(batch_loss, input.size(0))
            top1.update(prec1.item(), input.size(0))
            top5.update(prec5.item(), input.size(0))
            train_samples += label.numel()
            train_acc += (total_output.argmax(1) == label).float().sum().item()
            functional.reset_net(net)

            # time
            batch_time.update(time.time() - end)
            end = time.time()

            # plot progress
            bar.suffix  = '({batch}/{size}) Data: {data:.3f}s | Batch: {bt:.3f}s | Total: {total:} | ETA: {eta:} | Loss: {loss:.4f} | top1: {top1: .4f} | top5: {top5: .4f}'.format(
                        batch=batch_idx,
                        size=len(train_data_loader),
                        data=data_time.avg,
                        bt=batch_time.avg,
                        total=bar.elapsed_td,
                        eta=bar.eta_td,
                        loss=losses.avg,
                        top1=top1.avg,
                        top5=top5.avg,
                        )
            bar.next()
        bar.finish()
        train_loss /= train_samples
        train_acc /= train_samples
        writer.add_scalar('train_loss', train_loss, epoch)
        writer.add_scalar('train_acc', train_acc, epoch)
        lr_scheduler.step()

        # testing
        net.eval()
        batch_time = AverageMeter()
        data_time = AverageMeter()
        losses = AverageMeter()
        top1 = AverageMeter()
        top5 = AverageMeter()
        end = time.time()
        bar = Bar('Testing', max=len(test_data_loader))
        test_loss = 0
        test_acc = 0
        test_samples = 0
        batch_idx = 0
        with torch.no_grad():
            for frame, label in test_data_loader:
                batch_idx += 1
                if args.dataset != 'dvscifar10':
                    frame = frame.float().cuda()
                    if args.dataset == 'dvsgesture':
                        frame = frame.transpose(0, 1)
                label = label.cuda()
                t_step = args.T
                total_loss = 0
                for t in range(t_step):
                    if args.dataset == 'dvscifar10':
                        input = frame[t].float().cuda()
                    elif args.dataset == 'dvsgesture':
                        input = frame[t]
                    else:
                        input = frame
                    if t == 0:
                        output = net(input)
                        total_output = output.clone().detach()
                    else:
                        output = net(input)
                        total_output += output.clone().detach()
                    if args.loss_lambda > 0.0:
                        if args.mse_n_reg:
                            label_one_hot = F.one_hot(label, num_classes).float()
                        else:
                            label_one_hot = torch.zeros_like(output).fill_(args.loss_means).to(output.device)
                        mse_loss = criterion_mse(output, label_one_hot)
                        loss = ((1 - args.loss_lambda) * F.cross_entropy(output, label) + args.loss_lambda * mse_loss) / t_step
                    else:
                        loss = F.cross_entropy(output, label) / t_step
                    total_loss += loss
                test_samples += label.numel()
                test_loss += total_loss.item() * label.numel()
                test_acc += (total_output.argmax(1) == label).float().sum().item()
                functional.reset_net(net)

                # accuracy and loss
                prec1, prec5 = accuracy(total_output.data, label.data, topk=(1, 5))
                losses.update(total_loss, input.size(0))
                top1.update(prec1.item(), input.size(0))
                top5.update(prec5.item(), input.size(0))

                # time
                batch_time.update(time.time() - end)
                end = time.time()

                # plot progress
                bar.suffix  = '({batch}/{size}) Data: {data:.3f}s | Batch: {bt:.3f}s | Total: {total:} | ETA: {eta:} | Loss: {loss:.4f} | top1: {top1: .4f} | top5: {top5: .4f}'.format(
                            batch=batch_idx,
                            size=len(test_data_loader),
                            data=data_time.avg,
                            bt=batch_time.avg,
                            total=bar.elapsed_td,
                            eta=bar.eta_td,
                            loss=losses.avg,
                            top1=top1.avg,
                            top5=top5.avg,
                            )
                bar.next()
        bar.finish()
        test_loss /= test_samples
        test_acc /= test_samples
        writer.add_scalar('test_loss', test_loss, epoch)
        writer.add_scalar('test_acc', test_acc, epoch)

        # saving checkpoint
        save_max = False
        if test_acc > max_test_acc:
            max_test_acc = test_acc
            save_max = True
        checkpoint = {
            'net': net.state_dict(),
            'optimizer': optimizer.state_dict(),
            'lr_scheduler': lr_scheduler.state_dict(),
            'epoch': epoch,
            'max_test_acc': max_test_acc,
        }
        if save_max:
            torch.save(checkpoint, os.path.join(out_dir, 'checkpoint_max.pth'))
        torch.save(checkpoint, os.path.join(out_dir, 'checkpoint_latest.pth'))
        total_time = time.time() - start_time
        print(f'epoch={epoch}, train_loss={train_loss}, train_acc={train_acc}, test_loss={test_loss}, test_acc={test_acc}, max_test_acc={max_test_acc}, total_time={total_time}, escape_time={(datetime.datetime.now()+datetime.timedelta(seconds=total_time * (args.epochs - epoch))).strftime("%Y-%m-%d %H:%M:%S")}')
        print("after one epoch: %fGB" % (torch.cuda.max_memory_cached(0) / 1024 / 1024 / 1024))

def is_path(s):
    return os.path.exists(s) and s.endswith('.pth')

if __name__ == '__main__':
    main()
